/*******************************************************************************
*  (c) synertronixx GmbH
*  Vahrenwalder Str. 7
*  D-30165 Hannover
*  Tel. : 0511 / 93 57 670
*  Fax. : 0511 / 93 57 679
*  Web  : www.synertronixx.de
*  eMail: info@synertronixx.de
* ----------------------------
* Name          : can_test.c
* Version       : 1.02
* Datum         : 16.01.2004
* Autor         : Blaschke
* Beschreibung  : CAN2Web C-Treiber
*                 Beispielprogramm fuer die Verwendung der Treiberroutinen
*                 fuer den CAN-Bus
*                 Es werden N CAN-Telegramme ueber den CAN-Bus bei einer
*                 Baudrate von rate kBit/s gesendet und die eingehenden Daten
*                 ausgegeben.
* Programmaufruf: can_test [N] [rate] ; z.B. can_test 23 100
*                 Sonderfall: N ist negativ, Programm sendet keine Daten sondern
*                             zeigt alle eingehenden Meldungen an bis
*                             |N|-Meldungen empfangen wurden
* *****************************************************************************/

//Dos-Header
#include <DOS.H>
#include <STDIO.H>
#include <STRING.H>
#include <STDLIB.H>

// DeviLAN-Header
#include "../can2web_101/can.h"
#include "../can2web_101/intrhndl.h"


/******************************************************************************/
/*  CAN Message als Text ausgeben                                             */
/******************************************************************************/
void PrintCANMessageString(TCANMsg message)
{
  char str[255], str2[10];
  int nr;

  sprintf(str,"ID:%08lX ", message.ID);
  sprintf(str2,"RTR:%d ", message.RTR);
  strcat(str, str2);
  sprintf(str2,"Laenge:%d ", message.LEN);
  strcat(str, str2);

  if (message.RTR==0)
  { // Nur Daten ausgeben, wenn RTR Bit=0 da bei RTR=1 keine Daten vorhanden   
    for (nr=0;nr<=message.LEN-1;nr++)
    { // Datenbyte
      sprintf(str2, " %02X", message.DATA[nr]);
      strcat(str,str2);
    }
  }
  strcat(str,"\n");
  printf(str);
}


/*******************************************************************************
* H A U P T P R O G R A M M
*******************************************************************************/

int main(int argc, char *argv[])
{
   TCANMsg can_out; // CAN-Message-Struktur fuer Daten senden
   TCANMsg can_in;  // CAN-Message-Struktur fuer Daten empfangen

   unsigned int can_baudrate = 500;        // 100 kBit/s als Baudrate einstellen
                                           // weitere moegliche Werte sind:
                                           // 5, 10, 20, 50, 100, 125, 250, 500, 1000
   unsigned long int can_mask=0x1FFFFFFFL; // CAN-Mask-Register (0x00..0x7FF) fuer CAN 2.0A
                                           // und (0x00000000..0x1FFFFFFF)   fuer CAN 2.0B
   unsigned long int can_code=0x1FFFFFFFL; // CAN-Code-Register (0x00..0x7FF) fuer CAN 2.0A
                                           // und (0x00000000..0x1FFFFFFF)   fuer CAN 2.0B
   unsigned char     can_error;            // Fehlermeldung

   long int n=1;                            // Schleifenzaehler
   long int N=10;                           // Anzahl der ausgegebenen Zeichenketten
   int i;                                   // allgemeiner Zaehler
   int rate;                                // vorgegebene Baudrate
   int m;                                   // Warteschleifenzaehler
   long int n_old;                          // Pufferzaehler

   printf("Starte Testprogramm fuer den CAN-Bus\n\n");

   // Anzahl der Durchlaeufe bestimmen
   if (argc>=2)
     sscanf(argv[1],"%ld",&N);
   if (N<0)
   {
     n=2*N;
     printf("Warte auf eingehende Meldungen\n\n");
   }

   // Baudrate bestimmen
   if (argc>=3)
   {
     if (sscanf(argv[2],"%d",&rate)==1)
       can_baudrate=rate;
   }

   DeviLANInit();  // Allgemeine DeviLAN Initialisierungsfunktion
                   // muss zu Beginn des Programms einmal aufgerufen werden

   can_error = CANInit(can_baudrate, CAN_EXTENDED, can_mask, can_code);  // CAN-Bus initialisieren
   if (can_error!=CAN_INIT_OK)
   {
     printf("Fehler bei Initialisierung: %02X\n", can_error);
     printf("Programm wird abgebrochen!\n");
     exit (0);
   }

   IPCEnableExternalInterrupts(); // Externe Interrupts freigeben
   //IPCDisableExternalInterrupts();

   CANOn();                       // CAN-Bus einschalten, Interrupt freigeben

   //Hauptschleife
   do
   {
     // CAN-Daten abfragen und ausgeben
     if (CANValidData()!=0)
     { // Neue CAN-Meldung von CAN-Bus
       can_in=CANGetData();           // CAN-Meldung aus Puffer lesen
       printf("Empfangen: ");         // und
       PrintCANMessageString(can_in); // ausgeben
       if (N<0)
         n++;
     }


     // CAN-Daten senden und ausgeben
     if (N>0)
     {
       // Telegramm erzeugen
       can_out.ID  = n; //(long) (0x250+n-1);
       can_out.RTR = 0;
       can_out.LEN = 8;
       for (i=0;i<=can_out.LEN-1;i++)
         can_out.DATA[i]=(unsigned char) n;

       if (CANSendData(can_out)==0)           // CAN-Meldung senden
       {
         // ggf. CAN-Fehlermeldung ausgeben
         // printf("Sendepuffer voll\n\r");
       }
       else
       {
         // ggf. CAN-Meldung ausgeben
         //printf("Gesendet: ");           // und
         //PrintCANMessageString(can_out); // ausgeben
       }  
     }

     // Fehlerabfrage und Ausgabe
     can_error = CANGetStatus();
     if ( (can_error&0xFC) != CAN_ERR_OK)
     {
       printf("CAN-Fehler %02X: ", can_error);
       can_error&=0xFC;
       if (can_error&CAN_ERR_OVERRUN)              printf(" OVERRUN");
       if (can_error&CAN_ERR_BUSERROR)             printf(" BUSERROR");
       if (can_error&CAN_ERR_BUSOFF)               printf(" BUSOFF");
       if (can_error&CAN_ERR_RECEIVEBUF_OVERFLOW)  printf(" RECEIVEBUF_OVERFLOW");
       if (can_error&CAN_ERR_TRANSMITBUF_OVERFLOW)
       {
         n--;
         printf(" TRANSMITBUF_OVERFLOW");
       }
       printf("\n");
     }

     if (N>0)
       n++;
   }
   while(n<=N);

   // Warten bis Sendepuffer leer ist bzw.
   // bis max. 50 Wartedurchlaeufe vorueber, falls nicht gesendet wird
   m=50;
   n_old=0;
   do
   {
     n=CANTransmitData();
     if (n<n_old)
     {
       m++;
       n_old=n;
     }
     m--;
     printf("Anzahl der Meldungen im Sendepuffer:%d\n\r",n);
   }
   while( (n!=0) && (m>0) ); // Warten bis Sendepuffer leer ist;
                             // alle Meldungen auf CAN gelegt sind

   CANOff();            // CAN-Bus abschalten, Interrupt sperren
   CANRestoreOldInt();  // Wichtig: Alten Interruptvektor wieder herstellen !
                        // CANRestoreOldInt() muss vor dem Programmende einmal
                        // aufgerufen werden !

   printf("\n\rCAN-Testprogramm beendet\n\r");

   return 0;
}


