/*******************************************************************************
*  (c) synertronixx GmbH
*  Lange Laube Str. 22
*  30159 Hannover
*  Tel. : 0511 / 262 999  0
*  Fax. : 0511 / 262 999 29
*  Web  : www.synertronixx.de
*  eMail: devilanapp@synertronixx.de
* ----------------------------
* Project       : DeviLAN App RC server  
* Name          : udpconfig.c
* Version       : see variable "appl_version_info"
* Date          : 28.09.2012
* Author        : DeviLAN app team
* Description   : - Opens a UDP listen socket
*                 - waits for UPD broadcast identification requests
*                 - send identification messages
*
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the Free Software
*   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
* *****************************************************************************/

// system header
#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <asm/unistd.h>
#include <sys/select.h>         //for select, we need select
#include <arpa/inet.h>
#include <sys/wait.h>


// project header
#include "udpconfig.h"

// globals
int socket_nr;                   // just a counter 
fd_set udp_active_fd_set;        // bitmask for a set/group of file descritors
fd_set udp_read_fd_set;          // bitmasks for file descriptors
struct timeval time2;            // time structure for 'select'
int udp_running=1;               // main loop is running


int udp_listen_socket;           // UDP listen socket
struct udp_data my_udp_data;

// externals 
extern char* appl_name;          // Application name for debug messages
extern char* appl_version_info;  // Version information


/******************************************************************************
Function:     UDPCreateServerSocket  
Description:  creates a new socket
Parameter:    udp_broadcast_port: udp port for listeninge
Return:       sock: created socket number
Date:         21.02.2007
Changes:
 ******************************************************************************/
int
UDPCreateServerSocket (uint16_t udp_broadcast_port)
{
	int sock;
	struct sockaddr_in servername;
	size_t size = sizeof (servername);

	// Create the socket
	sock = socket (PF_INET, SOCK_DGRAM, 0);

	if (sock < 0)
	{
		perror ("UDPSocket create socket");
		exit (EXIT_FAILURE);
	}

	// Give the socket a name
	servername.sin_family = AF_INET;
	servername.sin_port = htons (udp_broadcast_port);
	servername.sin_addr.s_addr = htonl (INADDR_ANY);
	if (bind (sock, (struct sockaddr *) &servername, size) < 0)
	{
		perror ("UDPSocket bind");
		exit (EXIT_FAILURE);
	}

	return sock;
}


/******************************************************************************
Function:     UDPGetConfigData 
Description:  gets the udp-data stored in /etc/network/interfaces
              Remark: when your network settings are stored somewhere else
                       set in this function the correct values
Parameter:    *new_udp_data: pointer to structure to save the read data
Return:       ERR_FILE_NOTFOUND
              ERR_INTERFACE_NOTFOUND
              ERR_INETDATA_NOTFOUND
              ERR_UDPDATA_NOTFOUND
              0: ok
Date:         09.10.2012
Changes:
 ******************************************************************************/
int
UDPGetConfigData (struct udp_data *new_udp_data)
{
	FILE *htm;
	int j = 0;
	char buffer[1000] = "\0";
	char part1[1000] = "\0", part2[1000] = "\0";
	char databuffer[33] = "\0";
	char *pointer;
	int res = 0, err = 0;

	/* get udp-data and inet-data from file */
	htm = fopen ("/etc/network/interfaces", "r");
	if (!htm)
		return ERR_FILE_NOTFOUND;

	while (!feof(htm))
	{
		buffer[j++] = (char)fgetc (htm);
	}
	fclose (htm);

	/* divide string into 2 parts (part2: useful data) */
	pointer = strstr (buffer, "iface eth0 inet static\n");
	if (!pointer)
		return ERR_INTERFACE_NOTFOUND;

	strncpy (part1, buffer, pointer - buffer + 23);
	strncpy (part2, pointer + 23, strlen (buffer) - strlen (part1)); 

	// searches for udp data and puts data into the udp-structure
	res = string_cut (part2, databuffer, "address");
	if (res == ERR_SEARCHSTRING_NOTFOUND)
		return ERR_INETDATA_NOTFOUND;
	new_udp_data->inet.address = inet_addr (databuffer);

	res = string_cut (part2, databuffer, "netmask");
	if (res == ERR_SEARCHSTRING_NOTFOUND)
		return ERR_INETDATA_NOTFOUND;
	new_udp_data->inet.netmask = inet_addr (databuffer);

	res = string_cut (part2, databuffer, "gateway");
	if (res == ERR_SEARCHSTRING_NOTFOUND)
		return ERR_INETDATA_NOTFOUND;
	new_udp_data->inet.gateway = inet_addr (databuffer);


	// some default data
    new_udp_data->udp.serial           = DEFAULT_UDP_SERIAL; 
    strcpy (new_udp_data->udp.modul_name, DEFAULT_UDP_MODULNAME);
    new_udp_data->udp.tcp_ip_port      = DEFAULT_UDP_TCPPORT;
    new_udp_data->udp.udp_port         = DEFAULT_UDP_UDPPORT;
	new_udp_data->inet.dhcp            = DEFAULT_INET_DHCP;
	new_udp_data->udp.udp_command      = UDP_COMMAND_INDENTIFICATION;
	new_udp_data->udp.udp_datagram_len = UDP_INDENT_DATAGRAM_LEN;

	if (err)
		return ERR_UDPDATA_NOTFOUND;
	else
		return 0;
}


/******************************************************************************
Function:     UDPCreateIndentMsg 
Description:  creates a new message to send back after broadcast or indent-cmd
Parameter:    my_udp_data: udp-data structure with data sources
 *message: pointer to the message being ccreated
Return:       none
Date:         21.02.2007
Changes:
 ******************************************************************************/
void
UDPCreateIndentMsg (struct udp_data new_udp_data, char *message)
{
	int i;
	int m = 0;

	for (i = 1; i >= 0; i--)
		message[m++] = new_udp_data.udp.udp_command >> (i * 8);

	for (i = 0; i < 6; i++)
		message[m++] = new_udp_data.inet.mac_nr[i];

	for (i = 1; i >= 0; i--)
		message[m++] = new_udp_data.udp.udp_datagram_len >> (i * 8);

	message[m++] = new_udp_data.inet.dhcp;

	for (i = 0; i < 4; i++)
		message[m++] = new_udp_data.inet.address >> (i * 8);

	for (i = 1; i >= 0; i--)
		message[m++] = new_udp_data.udp.tcp_ip_port >> (i * 8);

	for (i = 0; i < 4; i++)
		message[m++] = new_udp_data.inet.netmask >> (i * 8);

	for (i = 0; i < 4; i++)
		message[m++] = new_udp_data.inet.gateway >> (i * 8);

	for (i = 0; i < 33; i++)
		message[m++] = new_udp_data.udp.modul_name[i];

	for (i = 3; i >= 0; i--)
		message[m++] = new_udp_data.udp.serial >> (i * 8);

}


/******************************************************************************
Function:     UDPReceiveHandler  
Description:  handles the new data when received
Parameter:    sock: active socket number
Return:       ERR_INVALIDCOMMAND
ERR_NOTMYDATAGRAM
ERR_FILENOTFOUND
ERR_INTERFACENOTFOUND
0: ok
Date:         09.10.2012
Changes:
 ******************************************************************************/
int
UDPReceiveHandler (int sock, struct udp_data *my_udp_data)
{
	struct sockaddr_in clientname;
	char message[UDP_MAXLEN];
	int datagram_len;
	struct udp_data new_udp_data;
	size_t size = sizeof(clientname);
	int res = 0;

	/* get udp- and inet-data */
	res = UDPGetConfigData (&new_udp_data);
	if ((res == ERR_FILE_NOTFOUND) || (res == ERR_INTERFACE_NOTFOUND)  || \
						(res == ERR_INETDATA_NOTFOUND))
		return res;

    // get the MAC
    GetMAC(sock, &new_udp_data);

    // read data from socket 
    datagram_len = recvfrom (sock, message, UDP_MAXLEN, 0,(struct sockaddr *) \
							&clientname, &size);

	if (datagram_len < 0)
	{
		perror ("UDPSocket socket read");
		udp_running=0;
	}

	/* handle datagram */
	if (datagram_len + 1 < 60)
	{
		int k = 0, i;
		int command_id, command_len, wrong_mac = 0;
		char sendbuffer[128];

		/* get command and mac from datagram */
		command_id = (message[k++] << 8);
		command_id |= message[k++];

		for (i = 0; i < 6; i++)
		{
			if (my_udp_data->inet.mac_nr[i] != message[k++])
				wrong_mac = 1;
		}

		command_len = (message[k++] << 8);
		command_len |= message[k++];

		/* handle command */
		switch (command_id)
		{
			case UDP_COMMAND_INDENTIFY_ALL: // ask all modules for identification message
				if ((command_len == 10) && (datagram_len == 10))
				{ // create answers and send it to socket
					UDPCreateIndentMsg (new_udp_data, sendbuffer);
					sendto (sock, sendbuffer, UDP_MAXLEN, 0, (struct sockaddr *) &clientname, size);
				}
				else
				{
					return ERR_NOTMYDATAGRAM;
				}
				break;

            // Remark: all other commands are not processed here

			default:
				return ERR_INVALIDCOMMAND;
				break;
		}
	}

	return 0;
}


/******************************************************************************
Function:     string_cut  
Description:  cuts out data between the search-string and end-of-line from
the input-string
Parameter:    *source: source string to divide 
 *data: destination string
 *string: search string
Return:       ERR_SEARCHSTRING_NOTFOUND
0: ok
Date:         21.02.2007
Changes:
 ******************************************************************************/
int
string_cut (char *source, char *data, char *string)
{
	char *pointer;
	char *start_pointer;
	int string_len;

	/* search for string */
	pointer = strstr (source, string);
	if (!pointer)
		return ERR_SEARCHSTRING_NOTFOUND;

	/* get data after string */
	start_pointer = pointer + strlen (string) + 1;
	string_len = strchr (pointer, '\n') - pointer - strlen (string) - 1;
	strncpy (data, start_pointer, string_len);
	data[string_len] = '\0';

	return 0;
}


/******************************************************************************
Function:     GetMAC  
Description:  writes MAC-address to a pointer to an udp-data structure
Parameter:    sock:         active socket number
              *my_udp_data: pointer to udp-data structure
Return:       none
Date:         09.10.2012
Changes:
 ******************************************************************************/
void
GetMAC(int sock, struct udp_data *my_udp_data)
{
	struct ifreq ifr;
	struct ifreq *IFR;
	struct ifconf ifc;
	char buf[1024];
	int i;

	/* fill ifc with all active interfaces */
	ifc.ifc_len = sizeof(buf);
	ifc.ifc_buf = buf;
	ioctl(sock, SIOCGIFCONF, &ifc);

	/* search for correct interface and get mac address */
	IFR = ifc.ifc_req;
	for (i = ifc.ifc_len / sizeof(struct ifreq); --i >= 0; IFR++)
	{
		strcpy(ifr.ifr_name, IFR->ifr_name);
		if (ioctl(sock, SIOCGIFFLAGS, &ifr) == 0)
		{
			if (! (ifr.ifr_flags & IFF_LOOPBACK))
			{
				ioctl(sock, SIOCGIFHWADDR, &ifr);
				bcopy(ifr.ifr_hwaddr.sa_data,
						my_udp_data->inet.mac_nr, 6);

				/*printf("MAC:");
				  for(i=0;i<6;i++)
				  printf(" %02X", my_udp_data->inet.mac_nr[i]);
				  printf("\n\n");*/
				break;
			}
		}
	}
}

/******************************************************************************
Function:     UDPInit  
Description:  init UDP communication for identification message
Parameter:    none
Return:       -1: on error
               0: otherwise
Date:         09.10.2012
Changes:
 ******************************************************************************/
int
UDPInit(void)
{
    int udp_result;

    printf("%s: Start UDP socket\n", appl_name);

	/* timer settings for select*/
	time2.tv_sec  = 0;
	time2.tv_usec = 0;

	/* get udp- and inet-data from file and other sources */
	udp_result = UDPGetConfigData(&my_udp_data);
	if (udp_result!=0)
	{ // an error occured
		if (udp_result == ERR_FILE_NOTFOUND)
			fprintf(stderr, "%s: config-file not found!\n", appl_name);
		if (udp_result == ERR_INTERFACE_NOTFOUND)
			fprintf(stderr, "%s: interface not found! not connected!\n", appl_name);
		if (udp_result == ERR_INETDATA_NOTFOUND)
			fprintf(stderr, "%s: invalid inet-data!\n", appl_name);
		if (udp_result != ERR_UDPDATA_NOTFOUND)
		{
			shutdown(udp_listen_socket, SHUT_RDWR);
			return -1;
		}
	}

	/* make udp-server socket */
	udp_listen_socket = UDPCreateServerSocket (my_udp_data.udp.udp_port);

	/* get mac-nr from socket */
	GetMAC (udp_listen_socket, &my_udp_data);
 
	/* Initialize the set of active sockets. */
	FD_ZERO (&udp_active_fd_set);                     // clear all bitmask for file descriptors
	FD_SET (udp_listen_socket, &udp_active_fd_set);   // set bit for file descriptor 

    return 0;
}

/******************************************************************************
Function:     UDPProcessCommands  
Description:  process incomming UDP messages
Parameter:    none
Return:       none
Date:         09.10.2012
Changes:
 ******************************************************************************/
void
UDPProcessCommands(void)
{
  int udp_result;

  // Check sockeks for communication
  udp_read_fd_set = udp_active_fd_set;     // copy for select function
  if (select (FD_SETSIZE, &udp_read_fd_set, NULL, NULL, &time2) < 0)
  {
      perror ("UDP-Socket select"); 
      udp_running=0;
  }

  // handle all sockets 
  for (socket_nr = 0; socket_nr < FD_SETSIZE; ++socket_nr)
  {
      if (FD_ISSET (socket_nr, &udp_read_fd_set))
      {
          udp_result = UDPReceiveHandler(udp_listen_socket, &my_udp_data);
          if (udp_result!=0)
          {
              if (udp_result == ERR_FILE_NOTFOUND){
                  fprintf(stderr, "%s: config-file not found!\n", appl_name);
                  udp_running = 0;
              }
              if (udp_result == ERR_INTERFACE_NOTFOUND){
                  fprintf(stderr, "%s: interface not found! not connected!\n", appl_name);
                  udp_running = 0;
              }
              if (udp_result == ERR_INETDATA_NOTFOUND){
                  fprintf(stderr, "%s: invalid inet-data!\n", appl_name);
                  udp_running = 0;
              }
          }
          else{
              udp_running=0;
          }
      }
  }
}


/******************************************************************************
Function:     UDPClose  
Description:  close UDP listener
Parameter:    none
Return:       none
Date:         09.10.2012
Changes:
 ******************************************************************************/
void UDPClose(void)
{
  close(udp_listen_socket);
}
	
// end of file
