#!/usr/bin/python
#*******************************************************************************
#  synertronixx GmbH
#  Lange Laube Str. 22
#  30159 Hannover
#  Tel. : 0511 / 262 999  0
#  Fax. : 0511 / 262 999 29
#  Web  : www.synertronixx.de
#  eMail: devilanapp@synertronixx.de
# ----------------------------
# Project       : Raspberry Pi Remote Control to use with iOS App DeviLAN 
# Name          : devilan_pi_rc_server_piface.py
# IDLE          : version 2.7.3 , Python version 2.7.3 & Tk version 8.5                   
# Version       : 1.4
# Date          : 24.04.2014
# Author        : DeviLAN app team
# Description   :
#
# - This program is a TCP/IP server for Raspberry Pi modules with IO module "Pi Face".
#   Use it as example to create your own Remote Control GUI for iPhone, iPad and iPod
#   to control Raspberry Pi and peripheral devices.
#
# - Use iOS App DeviLAN (TCP/IP based Remote Control tool) to find and control
#   Raspberry Pis which you can download for free from the Appstore
#   http://itunes.apple.com/us/app/devilan/id544777502?l=de&ls=1&mt=8#
#
# - Use Android App DeviLAN (TCP/IP based Remote Control tool) to find and control
#   Raspberry Pis which you can download for free from Google play
#   https://play.google.com/store/apps/details?id=com.synertronixx.devilan
#
# - Get further information about DeviLAN app from our website:
#   http://www.synertronixx.de/produkte/devilanapp/DeviLANApp_search_configure.htm
#   http://www.synertronixx.de/produkte/devilanapp/DeviLANApp_remote_control.htm
#
# - Testing:
#   1. Copy this file to "/home/pi/DeviLAN_Pi_Servers/"
#   2. Start Python IDLE
#   3. Open this file
#   4. Run the script (F5)
#
# - Start script on power up (boot) automatically:
#   1. Open file  "/etc/rc.local" in editor
#      e.g.: "sudo nano /etc/rc.local"
#   2. Add line "python /home/pi/DeviLAN_Pi_Servers/devilan_pi_rc_server_piface.py &
#      and store the file
#   3. Make file executable
#      e.g.: "chmod 755 /home/pi/DeviLAN_Pi_Servers/devilan_pi_rc_server_piface.py
#
# - History:
#   17.03.2014: Released first beta version (quick & dirty)
#               todo: split menus and data handling in several files
#   24.03.2014: Added load file from Rasbperry Pi webserver for webview
#   25.03.2014: Added more examples for GUIs
#   10.04.2014: Bugfix in menu
#   24.04.2014: New Google play links for Android App 
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
# *****************************************************************************/

import socket
import sys
import fcntl
import struct
from thread import *
from time import sleep
import time
import pifacedigitalio


APPNAME = "DeviLAN-Pi RC-Server PiFace 1.4: " # application name for debug output

INTERFACE = "eth0"  # LAN interface
HOST = ""   # Symbolic name meaning all available interfaces
PORT = 3333 # Communication port for listening socket
DELAY = 0.2 # a delay in seconds
WEBSERVER_PORT = 8080  # Webserver communication port

menuno = 0  # GUI menu no.
pifacedigital = pifacedigitalio.PiFaceDigital() # interface to PiFace digital



#------------------------------------------------------------------------------
# Function sendMenu: send selected menu to DeviLAN app
#------------------------------------------------------------------------------
def sendMenu(conn):

    global menuno

    print (APPNAME + "Sending menu %d" % menuno )
    if menuno == 0: # Main
        sendMenuMain(conn)
        return
    if menuno == 1: # Switches & IO history (layout 1)
        sendMenu1(conn, 1)
        return
    if menuno == 2: # Switches & IO history (layout 2)
        sendMenu2(conn, 1)
        return
    if menuno == 3: # Relais Switches & Buttons & IO history
        sendMenu3(conn, 1)
        return
    if menuno == 4: # Slider for outputs & IO history
        sendMenu4(conn, 1)
        return

    # some more DeviLAN  app control elements
    if menuno == 5: # control elements button, switch, slider
        sendMenu5(conn, 1)
        return
    if menuno == 6: # control elements audio
        sendMenu6(conn, 1)
        return
    if menuno == 7: # control element map
        sendMenu7(conn, 1)
        return
    if menuno == 8: # control element webview local
        sendMenu8(conn, 1)
        return
    if menuno == 9: # control element webview internet
        sendMenu9(conn, 1)
        return



#------------------------------------------------------------------------------
# Function sendMenuMain: send menu main to DeviLAN app
#------------------------------------------------------------------------------
def sendMenuMain(conn):

    conn.send("rmt_clear\r") 
    sleep(DELAY)

    conn.send("rmt_header1_00_cfg ;Raspberry Pi + Pi Face Digital Main menu\r") 
    conn.send("rmt_text1_00_cfg ;Please choose a menu below. To return to main "
              "menu press button '<<Back to main menu'.;2\r") 

    conn.send("rmt_button1_m1_cfg ;1. Switches & IO history (layout 1)\r") 
    conn.send("rmt_button1_m2_cfg ;2. Switches & IO history (layout 2)\r") 
    conn.send("rmt_button1_m3_cfg ;3. Switches & Buttons & IO history for relais\r") 
    conn.send("rmt_button1_m4_cfg ;4. Slider for outputs & IO history\r") 

    conn.send("rmt_header1_01_cfg ;Examples for more GUI elements\r") 
    conn.send("rmt_button1_m5_cfg ;5. Control & Visualization elements\r") 
    conn.send("rmt_button1_m6_cfg ;6. Audio elements\r") 
    conn.send("rmt_button1_m7_cfg ;7. Visualization element Map\r") 
    conn.send("rmt_button1_m8_cfg ;8. Webview from Raspberry Pi\r") 
    conn.send("rmt_button1_m9_cfg ;9. Webview from internet\r") 
    conn.sendall("\r")

#------------------------------------------------------------------------------
# Function dataHandlerMenu1: handling incommimng commands for main menu
#------------------------------------------------------------------------------
def dataHandlerMenuMain(conn, parameterlist):

    global menuno
    parameter = parameterlist[0]

 
    # Menu button pressed
    if (parameter == "rmt_button1_m0_data"):
        value = int(parameterlist[1])
        if value == 0:
            menuno = 0
            sendMenu(conn)
        return
    if (parameter == "rmt_button1_m1_data"):
        value = int(parameterlist[1])
        if value == 0:
            menuno = 1
            sendMenu(conn)
        return
    if (parameter == "rmt_button1_m2_data"):
        value = int(parameterlist[1])
        if value == 0:
            menuno = 2
            sendMenu(conn)
        return
    if (parameter == "rmt_button1_m3_data"):
        value = int(parameterlist[1])
        if value == 0:
            menuno = 3
            sendMenu(conn)
        return
    if (parameter == "rmt_button1_m4_data"):
        value = int(parameterlist[1])
        if value == 0:
            menuno = 4
            sendMenu(conn)
        return
    if (parameter == "rmt_button1_m5_data"):
        value = int(parameterlist[1])
        if value == 0:
            menuno = 5
            sendMenu(conn)
        return
    if (parameter == "rmt_button1_m6_data"):
        value = int(parameterlist[1])
        if value == 0:
            menuno = 6
            sendMenu(conn)
        return
    if (parameter == "rmt_button1_m7_data"):
        value = int(parameterlist[1])
        if value == 0:
            menuno = 7
            sendMenu(conn)
        return
    if (parameter == "rmt_button1_m8_data"):
        value = int(parameterlist[1])
        if value == 0:
            menuno = 8
            sendMenu(conn)
        return
    if (parameter == "rmt_button1_m9_data"):
        value = int(parameterlist[1])
        if value == 0:
            menuno = 9
            sendMenu(conn)
        return




#------------------------------------------------------------------------------
# Function sendMenu1: send menu 1 to DeviLAN app
#------------------------------------------------------------------------------
def sendMenu1(conn, clear):

    # Read data from Pi Face
    output_value = pifacedigital.output_port.value
    input_value = pifacedigital.input_port.value

    if clear==1:
        conn.send("rmt_clear\r") 
        sleep(DELAY)

    conn.send("rmt_header1_00_cfg ;Control Menu 1\r") 
    conn.send("rmt_button1_m0_cfg ;<<Back to main menu\r") 
    conn.send("rmt_switch1_00_cfg ; Output 0, Relais; %d ;\r" % (output_value&0x01 == 0x01) )
    conn.send("rmt_switch1_01_cfg ; Output 1, Relais; %d ;\r" % (output_value&0x02 == 0x02) )
    conn.send("rmt_switch1_02_cfg ; Output 2; %d ;\r" % (output_value&0x04 == 0x04) )
    conn.send("rmt_switch1_03_cfg ; Output 3; %d ;\r" % (output_value&0x08 == 0x08) )
    conn.send("rmt_switch1_04_cfg ; Output 4; %d ;\r" % (output_value&0x10 == 0x10) )
    conn.send("rmt_switch1_05_cfg ; Output 5; %d ;\r" % (output_value&0x20 == 0x20) )
    conn.send("rmt_switch1_06_cfg ; Output 6; %d ;\r" % (output_value&0x40 == 0x40) )
    conn.send("rmt_switch1_07_cfg ; Output 7; %d ;\r" % (output_value&0x80 == 0x80) )

    if clear==1:
        conn.send("rmt_graph1_00_cfg ;Output history:;8;140; 35; %.1f; -0.5; 8.5; 9\r")
    conn.send("rmt_text1_00_cfg ;  Inputs = 0x%02X ; 0\r" % input_value)
    if clear==1:
        conn.send("rmt_graph1_01_cfg ;Input history:;8;140; 35; %.1f; -0.5; 8.5; 9\r")

    sendOutputToGraph(conn)
    sendInputToGraph(conn)
    conn.sendall("\r")


#------------------------------------------------------------------------------
# Function dataHandlerMenu1: handling incommimng commands for menu 1
#------------------------------------------------------------------------------
def dataHandlerMenu1(conn, parameterlist):
    parameter = parameterlist[0]


    if (parameter == "rmt_switch1_00_data"):
        valueled = int(parameterlist[1]) 
        print (APPNAME + " LED 0 value: %d" % valueled)
        pifacedigital.leds[0].value =  valueled
        sendOutputToGraph(conn)
        return

    if (parameter == "rmt_switch1_01_data"):
        valueled = int(parameterlist[1]) 
        pifacedigital.leds[1].value =  valueled
        return

    if (parameter == "rmt_switch1_02_data"):
        valueled = int(parameterlist[1]) 
        pifacedigital.leds[2].value =  valueled
        sendOutputToGraph(conn)
        return

    if (parameter == "rmt_switch1_03_data"):
        valueled = int(parameterlist[1]) 
        pifacedigital.leds[3].value =  valueled
        sendOutputToGraph(conn)
        return

    if (parameter == "rmt_switch1_04_data"):
        valueled = int(parameterlist[1]) 
        pifacedigital.leds[4].value =  valueled
        sendOutputToGraph(conn)
        return

    if (parameter == "rmt_switch1_05_data"):
        valueled = int(parameterlist[1]) 
        pifacedigital.leds[5].value =  valueled
        sendOutputToGraph(conn)
        return

    if (parameter == "rmt_switch1_06_data"):
        valueled = int(parameterlist[1]) 
        pifacedigital.leds[6].value =  valueled
        sendOutputToGraph(conn)
        return

    if (parameter == "rmt_switch1_07_data"):
        valueled = int(parameterlist[1]) 
        pifacedigital.leds[7].value =  valueled
        sendOutputToGraph(conn)
        return


#------------------------------------------------------------------------------
# Function sendMenu2: send menu 2 to DeviLAN app
#------------------------------------------------------------------------------
def sendMenu2(conn, clear):

    # Read data from Pi Face
    output_value = pifacedigital.output_port.value
    input_value = pifacedigital.input_port.value

    if clear==1:
        conn.send("rmt_clear\r") 
        sleep(DELAY)

    conn.send("rmt_header1_00_cfg ;Control Menu 2\r") 
    conn.send("rmt_button1_m0_cfg ;<<Back to main menu\r") 

    cfg_str = "rmt_switch3_00_cfg ;"
    cfg_str += ("Output 0, Relais; %d ;" % (output_value&0x01 == 0x01))
    cfg_str += ("Output 1, Relais; %d ;" % (output_value&0x02 == 0x02))
    cfg_str += ("Output 2; %d \r" % (output_value&0x04 == 0x04))
    conn.send(cfg_str)
    cfg_str = "rmt_switch3_01_cfg ;"
    cfg_str += ("Output 3; %d ;" % (output_value&0x08 == 0x08))
    cfg_str += ("Output 4; %d ;" % (output_value&0x10 == 0x10))
    cfg_str += ("Output 5; %d \r" % (output_value&0x20 == 0x20))
    conn.send(cfg_str)
    cfg_str = "rmt_switch3_02_cfg ;"
    cfg_str += ("Output 6; %d ;" % (output_value&0x40 == 0x40))
    cfg_str += ("Output 7; %d ;" % (output_value&0x80 == 0x80))
    cfg_str += (";\r")
    conn.send(cfg_str)

    if clear==1:
        conn.send("rmt_graph1_00_cfg ;Output history:;8;140; 35; %.1f; -0.5; 8.5; 9\r")
    conn.send("rmt_text1_00_cfg ;  Inputs = 0x%02X ; 0\r" % input_value)
    if clear==1:
        conn.send("rmt_graph1_01_cfg ;Input history:;8;140; 35; %.1f; -0.5; 8.5; 9\r")

    sendOutputToGraph(conn)
    sendInputToGraph(conn)
    conn.sendall("\r")

#------------------------------------------------------------------------------
# Function dataHandlerMenu2: handling incommimng commands for menu 2
#------------------------------------------------------------------------------
def dataHandlerMenu2(conn, parameterlist):
    parameter = parameterlist[0]

    if (parameter == "rmt_switch3_00_data_1"):
        valueled = int(parameterlist[1]) 
        pifacedigital.leds[0].value =  valueled
        sendOutputToGraph(conn)
        return
    if (parameter == "rmt_switch3_00_data_2"):
        valueled = int(parameterlist[1]) 
        pifacedigital.leds[1].value =  valueled
        sendOutputToGraph(conn)
        return
    if (parameter == "rmt_switch3_00_data_2"):
        valueled = int(parameterlist[1]) 
        pifacedigital.leds[2].value =  valueled
        sendOutputToGraph(conn)
        return
    if (parameter == "rmt_switch3_01_data_1"):
        valueled = int(parameterlist[1]) 
        pifacedigital.leds[3].value =  valueled
        sendOutputToGraph(conn)
        return
    if (parameter == "rmt_switch3_01_data_2"):
        valueled = int(parameterlist[1]) 
        pifacedigital.leds[4].value =  valueled
        sendOutputToGraph(conn)
        return
    if (parameter == "rmt_switch3_01_data_3"):
        valueled = int(parameterlist[1]) 
        pifacedigital.leds[5].value =  valueled
        sendOutputToGraph(conn)
        return
    if (parameter == "rmt_switch3_02_data_1"):
        valueled = int(parameterlist[1]) 
        pifacedigital.leds[6].value =  valueled
        sendOutputToGraph(conn)
        return
    if (parameter == "rmt_switch3_02_data_2"):
        valueled = int(parameterlist[1]) 
        pifacedigital.leds[7].value =  valueled
        sendOutputToGraph(conn)
        return



#------------------------------------------------------------------------------
# Function sendMenu3: send menu 3 to DeviLAN app
#------------------------------------------------------------------------------
def sendMenu3(conn, clear):

    # Read data from Pi Face
    output_value = pifacedigital.output_port.value
    input_value = pifacedigital.input_port.value

    if clear==1:
        conn.send("rmt_clear\r") 
        sleep(DELAY)

    conn.send("rmt_header1_00_cfg ;Control Menu 3\r") 
    conn.send("rmt_button1_m0_cfg ;<<Back to main menu\r") 

    cfg_str = "rmt_switch3_04_cfg ;"
    cfg_str += ("Output 0; %d ;" % (output_value&0x01 == 0x01))
    cfg_str += (";;")
    cfg_str += ("Output 1; %d \r" % (output_value&0x02 == 0x02))
    conn.send(cfg_str)
    conn.send("rmt_button3_04_cfg ;Output 0;;Output 1;\r")

    if clear==1:
        conn.send("rmt_graph1_00_cfg ;Output history:;2;140; 35; %.1f; -0.5; 2.5; 3; 1\r")

    conn.send("rmt_text1_00_cfg ;  Inputs = 0x%02X ; 0\r" % input_value)
    if clear==1:
        conn.send("rmt_graph1_01_cfg ;Input history:;8;140; 35; %.1f; -0.5; 8.5; 9\r")

    sendOutputToGraph(conn)
    if clear==1:
        sendInputToGraph(conn)
    conn.sendall("\r")


#------------------------------------------------------------------------------
# Function dataHandlerMenu3: handling incommimng commands for menu 3
#------------------------------------------------------------------------------
def dataHandlerMenu3(conn, parameterlist):
# Menu 3 data handling
    parameter = parameterlist[0]

    if (parameter == "rmt_switch3_04_data_1"):
        valueled = int(parameterlist[1]) 
        pifacedigital.leds[0].value =  valueled
        sendOutputToGraph(conn)
        return
    if (parameter == "rmt_switch3_04_data_3"):
        valueled = int(parameterlist[1]) 
        pifacedigital.leds[1].value =  valueled
        sendOutputToGraph(conn)
        return
    if (parameter == "rmt_button3_04_data_1"):
        valueled = int(parameterlist[1]) 
        pifacedigital.leds[0].value =  valueled
        sendMenu3(conn, 0)
        return
    if (parameter == "rmt_button3_04_data_3"):
        valueled = int(parameterlist[1]) 
        pifacedigital.leds[1].value =  valueled
        sendMenu3(conn, 0)
        return


#------------------------------------------------------------------------------
# Function sendMenu4: send menu 4 to DeviLAN app
#------------------------------------------------------------------------------
def sendMenu4(conn, clear):

    # Read data from Pi Face
    output_value = pifacedigital.output_port.value
    input_value = pifacedigital.input_port.value

    if clear==1:
        conn.send("rmt_clear\r") 
        sleep(DELAY)

    conn.send("rmt_header1_00_cfg ;Control Menu 3\r") 
    conn.send("rmt_button1_m0_cfg ;<<Back to main menu\r") 


    conn.send("rmt_slider1_01_cfg ; Pi-Face Output ; 0 ; 255 ; %d , %%d ; 1 \r" % output_value)

    if clear==1:
        conn.send("rmt_graph1_00_cfg ;Output history:;8;140; 35; %.1f; -0.5; 8.5; 9\r")

    conn.send("rmt_text1_00_cfg ;  Inputs = 0x%02X ; 0\r" % input_value)
    if clear==1:
        conn.send("rmt_graph1_01_cfg ;Input history:;8;140; 35; %.1f; -0.5; 8.5; 9\r")

    sendOutputToGraph(conn)
    if clear==1:
        sendInputToGraph(conn)

    conn.sendall("\r")

#------------------------------------------------------------------------------
# Function dataHandlerMenu4: handling incommimng commands for menu 1
#------------------------------------------------------------------------------
def dataHandlerMenu4(conn, parameterlist):
    parameter = parameterlist[0]


    if (parameter == "rmt_slider1_01_data"):
        value = int(parameterlist[1]) 
        #print (APPNAME + " Slider value: %d" % value)
        pifacedigital.output_port.value = value
        sendOutputToGraph(conn)
        return



#------------------------------------------------------------------------------
# Function sendMenu5: send menu 5 to DeviLAN app
#------------------------------------------------------------------------------
def sendMenu5(conn, clear):
    if clear==1:
        conn.send("rmt_clear\r") 
        sleep(DELAY)

    ipStr = (get_ip_address(INTERFACE))


    conn.send("rmt_header1_00_cfg ;Control Menu 5\r") 
    conn.send("rmt_button1_m0_cfg ;<< Back to main menu\r")
    conn.send("rmt_header1_aa_cfg ;All GUI Elements: Header type 1\r")
    
    # Control elements
    conn.send("rmt_header1_bb_cfg ;---- Control elements ----\r")
    conn.send("rmt_pick1_a1_cfg ;Picker description:;1;Value 1;Value 2;Value 3;Value 4;Value 5;Value 6\r")
    conn.send("rmt_radio1_a1_cfg ;Description for segment/radio button element:; 0;Value 1;Value 2;Value 3;Value n-1;Value n\r")
    conn.send("rmt_slider1_aa_cfg ;Slider type 1 description:; 0; 100; 25; %5.2fHz; 0.1\r")
    conn.send("rmt_switch1_aa_cfg ;Switch type 1 description:; 1\r")
    conn.send("rmt_switch3_aa_cfg ;Left; 1;Middle;1;Right;1\r")
    conn.send("rmt_button1_aa_cfg ;Middle Button type 1\r")
    conn.send("rmt_button2_aa_cfg ;Left Button type 2;Right Button type 2\r")
    conn.send("rmt_button3_aa_cfg ;Left Button;Type 3;Right Button\r")
    conn.send("rmt_edit1_aa_cfg ;Edit field type 1 description:;Enter text here\r")
        
    # Visualization elements
    conn.send("rmt_header1_cc_cfg ;---- Visualization elements ----\r")
    conn.send("rmt_graph1_aa_cfg ;Description text for graph:;4;170; 50; %.1f; -3.5; 3.5; 7\r")
    conn.send("rmt_progress1_aa_cfg ;Progress type 1 description:; 0; 100; 25; %5.2fHz; 1.0\r")
    conn.send("rmt_text1_aa_cfg ; Scroll text with speed 2, this could be a news ticker, your actual stock report, .....;2\r")

    # draw element
    conn.send("rmt_draw1_a1_cfg ;Description text for draw element:;150\r")
    conn.send("rmt_draw1_a1_data ;c;0.8;0.8;0.8;1\r")
    conn.send("rmt_draw1_a1_data ;r;0;0;1000;1000;1\r")
    conn.send("rmt_draw1_a1_data ;c;1;1;0;1\r")
    conn.send("rmt_draw1_a1_data ;a;140;60;50;0.0;6.3;0;1\r")
    conn.send("rmt_draw1_a1_data ;c;0;0;0;1\r")
    conn.send("rmt_draw1_a1_data ;a;120;40;5;0.0;6.3;0;1\r")
    conn.send("rmt_draw1_a1_data ;a;160;40;5;0.0;6.3;0;1\r")
    conn.send("rmt_draw1_a1_data ;a;140;60;30;0.0;3.14;0;0\r")
    conn.send("rmt_draw1_a1_data ;c;0;1;0;1\r")
    conn.send("rmt_draw1_a1_data ;t;20;120;20;l;Start your day with a smile!\r")

    conn.sendall("\r")
    


#------------------------------------------------------------------------------
# Function sendMenu6: send menu 6 to DeviLAN app
#------------------------------------------------------------------------------
def sendMenu6(conn, clear):
    if clear==1:
        conn.send("rmt_clear\r") 
        sleep(DELAY)

    ipStr = (get_ip_address(INTERFACE))


    conn.send("rmt_header1_00_cfg ;Control Menu 6\r") 
    conn.send("rmt_button1_m0_cfg ;<< Back to main menu\r")
    conn.send("rmt_header1_aa_cfg ;Audio GUI Elements:\r")
    
    # Audio elements
    # Audio from this Raspberry Pi, be sure python script devilan_pi_simple_webserver.py is running
    # so that audio file will be transmitted
    conn.send("rmt_header1_01_cfg ;Audio loaded from %s\r" % ipStr)
    conn.send("rmt_audio1_00_cfg ;Title: audio_01.m4a (%s);http://%s:%d/audio_01.m4a; 1\r"% (ipStr, ipStr, WEBSERVER_PORT))
    conn.send("rmt_audio1_01_cfg ;Title: audio_02.m4a (%s);http://%s:%d/audio_02.m4a; 1\r"% (ipStr, ipStr, WEBSERVER_PORT))
    conn.send("rmt_header1_02_cfg ;Audio loaded from synertronixx:\r")
    conn.send("rmt_audio1_02_cfg ;Title: Audio_01.m4a (www.synertronixx.de);http://www.synertronixx.de/audio/audio_01.m4a; 1\r")
    conn.send("rmt_audio1_03_cfg ;Title: Audio_02.m4a (www.synertronixx.de);http://www.synertronixx.de/audio/audio_02.m4a; 1\r")
    
    conn.sendall("\r")


#------------------------------------------------------------------------------
# Function sendMenu7: send menu 7 to DeviLAN app
#------------------------------------------------------------------------------
def sendMenu7(conn, clear):
    if clear==1:
        conn.send("rmt_clear\r") 
        sleep(DELAY)

    conn.send("rmt_header1_00_cfg ;Control Menu 7\r") 
    conn.send("rmt_button1_m0_cfg ;<< Back to main menu\r")
    conn.send("rmt_header1_01_cfg ;Map element\r")
    

    # map element
    conn.send("rmt_map1_aa_cfg ;Map description:; 200; 52.3768; 9.7283;1000; 0;1\r")
    conn.send("rmt_map1_aa_data ;52.3768; 9.7283;Homepage synertronixx;This is our homepage!;http://www.synertronixx.de\r")
    conn.send("rmt_map1_aa_data ;52.3768; 9.7285;Logo synertronixx;This in a our logo!;http://www.synertronixx.de/images/synertronixx_sphere.jpg\r")
    conn.send("rmt_map1_aa_data ;52.3768; 9.7287;SCB9324;This is a our socket computer SCB9324;http://www.synertronixx.de/images/scb9324_small_1.jpg\r")

    conn.sendall("\r")


#------------------------------------------------------------------------------
# Function sendMenu8: send menu 8 to DeviLAN app
#------------------------------------------------------------------------------
def sendMenu8(conn, clear):
    if clear==1:
        conn.send("rmt_clear\r") 
        sleep(DELAY)

    ipStr = (get_ip_address(INTERFACE))


    conn.send("rmt_header1_00_cfg ;Control Menu 8\r") 
    conn.send("rmt_button1_m0_cfg ;<< Back to main menu\r")
    
    # Webview from this Raspberry Pi, be sure python script devilan_pi_simple_webserver.py is running
    # so that HTML page and images will be transmitted
    conn.send("rmt_header1_01_cfg ;Webview from '%s'\r" % ipStr)
    conn.send("rmt_web1_00_cfg ;http://%s:%d/index.html;250\r" % (ipStr, WEBSERVER_PORT))

    conn.sendall("\r")


#------------------------------------------------------------------------------
# Function sendMenu9: send menu 9 to DeviLAN app
#------------------------------------------------------------------------------
def sendMenu9(conn, clear):
    if clear==1:
        conn.send("rmt_clear\r") 
        sleep(DELAY)

    conn.send("rmt_header1_00_cfg ;Control Menu 9\r") 
    conn.send("rmt_button1_m0_cfg ;<< Back to main menu\r")

    # Webview Internet
    conn.send("rmt_header1_01_cfg ;Webview 'www.synertronixx.de'\r")
    conn.send("rmt_web1_00_cfg ;http://www.synertronixx.de;250\r")



#------------------------------------------------------------------------------
# sendOutputToGraph, create values for graph from digital output
#------------------------------------------------------------------------------
def sendOutputToGraph(conn):
    # Read data from Pi Face
    output_value = pifacedigital.output_port.value

    # create "analog values" from digital outputs for graph
    v0 = 0 + (output_value&0x01 == 0x01) * 0.8
    v1 = 1 + (output_value&0x02 == 0x02) * 0.8 
    v2 = 2 + (output_value&0x04 == 0x04) * 0.8 
    v3 = 3 + (output_value&0x08 == 0x08) * 0.8 
    v4 = 4 + (output_value&0x10 == 0x10) * 0.8 
    v5 = 5 + (output_value&0x20 == 0x20) * 0.8 
    v6 = 6 + (output_value&0x40 == 0x40) * 0.8 
    v7 = 7 + (output_value&0x80 == 0x80) * 0.8 

    str_graph = "rmt_graph1_00_data ;%5.2f;%5.2f;%5.2f;%5.2f;%5.2f;%5.2f;%5.2f;%5.2f\r" % (v0,v1,v2,v3,v4,v5,v6,v7) 
    conn.send(str_graph)
    conn.send(str_graph)
    conn.send(str_graph)
   
#-------------------------------------------------------------------------------
# sendInputToGraph, create values for graph from digital inputs
#-------------------------------------------------------------------------------
def sendInputToGraph(conn):
    # Read data from Pi Face
    input_value = pifacedigital.input_port.value

    # create "analog values" from digital inputs for graph
    v0 = 0 + (input_value&0x01 == 0x01) * 0.8
    v1 = 1 + (input_value&0x02 == 0x02) * 0.8 
    v2 = 2 + (input_value&0x04 == 0x04) * 0.8 
    v3 = 3 + (input_value&0x08 == 0x08) * 0.8 
    v4 = 4 + (input_value&0x10 == 0x10) * 0.8 
    v5 = 5 + (input_value&0x20 == 0x20) * 0.8 
    v6 = 6 + (input_value&0x40 == 0x40) * 0.8 
    v7 = 7 + (input_value&0x80 == 0x80) * 0.8 

    str_graph = "rmt_graph1_01_data ;%5.2f;%5.2f;%5.2f;%5.2f;%5.2f;%5.2f;%5.2f;%5.2f\r" % (v0,v1,v2,v3,v4,v5,v6,v7) 
    conn.send(str_graph)
    conn.send(str_graph)
    conn.send(str_graph)

#--------------------------------------------------------------
# get_ip_address: get IP address for a given interface (ifname)
#--------------------------------------------------------------
def get_ip_address(ifname):
    s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    return socket.inet_ntoa(fcntl.ioctl(
        s.fileno(),
        0x8915,  # SIOCGIFADDR
        struct.pack('256s', ifname[:15])
    )[20:24])


#------------------------------------------------------------------------------
#------------------------------------------------------------------------------
# Function for handling connections. This will be used to create threads
#------------------------------------------------------------------------------
#------------------------------------------------------------------------------
def clientthread(conn):
    global menuno

    #Sending messages to connected client
    input_value_old = 0x00

    # Set socket options and timeouts for receive 
    conn.settimeout( 0.1)
    x = conn.getsockopt( socket.SOL_SOCKET, socket.SO_KEEPALIVE)
    if( x == 0):
        print (APPNAME + "Socket Keepalive off, turning on")
        x = conn.setsockopt( socket.SOL_SOCKET, socket.SO_KEEPALIVE, 1)
        print (APPNAME + "setsockopt=%s" %  x)
    else:
        print (APPNAME + "Socket Keepalive already on")

    menuno = 0

    # tell DeviLAN app we need GUI for remote control
    conn.send("modul_typ remote\r")
    sleep(DELAY)
    
    # Receiving from client 
    # Infinite loop so that function do not terminate and thread do not end.
    while True:
        try:
            data = conn.recv(1024)
 
        except socket.timeout:
            # print ("Socket timeout, loop and try recv() again")
            data = ""

            # read digital inputs
            input_value = pifacedigital.input_port.value
            if input_value != input_value_old:
                conn.send("rmt_text1_00_cfg ;  Inputs = 0x%02X ; 0\r" % input_value)
                input_value_old = input_value
                sendInputToGraph(conn)

            # time.sleep( 0.1)
            continue
 
        except:
            # traceback.print_exc()
            print (APPNAME + "Other Socket error, exit and try creating socket again")
            # break from loop
            break
 

        #get all command lines from buffer
        cmdslist =  data.split("\r")
        # print (APPNAME + "Commands: %d") % (len(cmdslist)-1)
        if len(cmdslist)== 1:
            print (APPNAME + "Error pipe broken ?? (%s)") % (cmdslist)
            # exit clientthread loop
            break
        
        x = 1
        for cmd in cmdslist:

            if(len(cmd) == 0 ):
                time.sleep( 0.1)
                break
            
            # print ("\Cmd no. %d: " % x)
            parameterlist = cmd.split(" ")
            parameter = parameterlist[0]

            print (APPNAME + "Handle '%s'" %  (parameter))
            if (parameter == "get_config"):
                sendMenu(conn)
                # break

            # always handle menu buttons
            dataHandlerMenuMain(conn, parameterlist)

            # Menu 1 data handling
            if menuno == 1:
                dataHandlerMenu1(conn, parameterlist)
                break

            # Menu 2 data handling
            if menuno == 2:
                dataHandlerMenu2(conn, parameterlist)
                break

            # Menu 3 data handling
            if menuno == 3:
                dataHandlerMenu3(conn, parameterlist)
                break

            # Menu 4 data handling
            if menuno == 4:
                dataHandlerMenu4(conn, parameterlist)
                break

            # Menu 5 data handling
            if menuno == 5:
                # dataHandlerMenu5(conn, parameterlist)
                # no data handling defined here just show the received comand
                print (APPNAME + "Command: '%s'" % cmd)
                break

            x = x + 1
            
     
    #came out of loop
    conn.close()
    print (APPNAME + "Server disconnected")


#------------------------------------------------------------------------------
# main
#------------------------------------------------------------------------------
def main():

    # Set up PiFace Digital
    pifacedigital = pifacedigitalio.PiFaceDigital()

    # Create listening socket
    listensocket = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    print (APPNAME + "Socket created")
     
    # Bind socket to local host and port
    try:
        listensocket.bind((HOST, PORT))
    except socket.error , msg:
            print (APPNAME + "Bind failed. Error Code : " + str(msg[0]) + " Message " + msg[1])
            sys.exit()
         
    print (APPNAME + "Socket bind complete")

    # Start listening on socket
    listensocket.listen(10)

    print (APPNAME + "Socket now listening at port %d" % PORT)


    # now keep talking with the client
    while 1:
        try:
            #wait to accept a connection - blocking call
            conn, addr = listensocket.accept()

            if (conn):
                print (APPNAME + "Connected with " + addr[0] + ":" + str(addr[1]))
                # start new thread takes 1st argument as a function name to be run,
                # second is the tuple of arguments to the function.
                start_new_thread(clientthread ,(conn,))


        except Exception as e:
            print (APPNAME + "Exception : %s" % str(e) )
            sys.exit(1)

    listensocket.close()


main()

# end of file

